package polymorphicTypes

// The Church encoding represents simple types as polymorphic types
// made from universal quantifications "forall t." and functions "A -> B".
// It can also encode existential quantification, and, more in general,
// inductive types.

// The Church encoding is related to the Yoneda lemma in category theory. 

object ChurchEncodingExample {
  
  // Product
  // (A * B) ~= forall C. (A -> B -> C) -> C
  case class Prod[A,B](a: A, b: B) {
    def elim[C](f: A=>B=>C): C = f(a)(b)
  }
  // projections
  def pi1[A,B](p: Prod[A,B]): A = p elim (a => _ => a)
  def pi2[A,B](p: Prod[A,B]): B = p elim (_ => b => b)
  
  // Sum
  // (A + B) ~= forall C. (A -> C) -> (B -> C) -> C
  trait Sum[A,B] {
    def elim[C](f: A=>C)(g: B=>C): C
  }
  // injections
  case class In1[A,B](a: A) extends Sum[A,B] {
    override def elim[C](f: A=>C)(g: B=>C): C = f(a)
  }
  case class In2[A,B](b: B) extends Sum[A,B] {
    override def elim[C](f: A=>C)(g: B=>C): C = g(b)
  }
  
  // 0 and 1 types
  // 0 ~= forall A. A
  trait Zero {
    def elim[A]: A
  }
  // 1 ~= forall A. A -> A
  case class One() {
    def elim[A](a: A): A = a
  }
  
  //===================================================
  // Examples
  
  // the isomorphism between 0 + A and A
  object Iso1 {
    def iso[A](s: Sum[Zero, A]): A = s.elim(z => z.elim)(a => a)
    def osi[A](a: A): Sum[Zero, A] = In2(a)
  }
  
  // the isomorphism between 1 * A and A
  object Iso2 {
    def iso[A](s: Prod[One, A]): A = s.elim(_ => a => a)
    def osi[A](a: A): Prod[One, A] = Prod(One(),a)
  }
  
  // distributivity of + and *
  // A*(B+C) ~= (A*B)+(A*C)
  object Distributivity1 {
    def iso[A,B,C](x: Prod[A,Sum[B,C]]): Sum[Prod[A,B],Prod[A,C]] =
      x.elim(a => s => 
        s.elim[Sum[Prod[A,B],Prod[A,C]]]
          (b => In1(Prod(a,b)))
          (c => In2(Prod(a,c))))
    def osi[A,B,C](x: Sum[Prod[A,B],Prod[A,C]]): Prod[A,Sum[B,C]] =
      x.elim[Prod[A,Sum[B,C]]](
          ab => ab.elim(a => b => Prod(a, In1(b))))(
          ac => ac.elim(a => c => Prod(a, In2(c))))
  }
  
  // distributivity of * and =>
  // (B*C)^A = (B^A)*(C^A)
  object Distributivity2 {
    def iso[A,B,C](f: A=>Prod[B,C]): Prod[A=>B,A=>C] =
      Prod(
        a => f(a).elim(b => _ => b),
        a => f(a).elim(_ => c => c))
    def osi[A,B,C](f: Prod[A=>B,A=>C]): A=>Prod[B,C] = a =>
      Prod(f.elim(x => _ => x)(a),
           f.elim(_ => y => y)(a))
  }
  
  // A power law
  // A^(B+C) = (A^B)*(A^C)
  object PowerLaw1 {
    def iso[A,B,C](f: Sum[B,C]=>A): Prod[B=>A,C=>A] =
      Prod(
        b => f(In1(b)),
        c => f(In2(c)))
    def osi[A,B,C](f: Prod[B=>A,C=>A]): Sum[B,C]=>A = s =>
      s.elim(
          b => f.elim(g => _ => g(b)))(
          c => f.elim(_ => h => h(c)))
  }

  //=====================================================
  
  // Existential encoding
  // exists a. f(a) ~= forall b. (forall a. f(a) -> b) -> b
  
  trait F[A] {
    // can be any arbitrary type
    def elim: Prod[A, A=>Int]
  }
  trait Helper[B] {
    def elim[A](f: F[A]): B 
  }
  trait ExistsF {
    def elim[B](f: Helper[B]): B
  }
  
  // In our specific example, f(a)=(a * a->Int), hence 
  // exists a. f(a) ~= Int
  object ExIso1 {
    def iso(e: ExistsF): Int = {
      object H extends Helper[Int] {
        override def elim[A](f: F[A]): Int =
          f.elim.elim(x => g => g(x))
      }
      e.elim[Int](H)
    }
    def osi(i: Int): ExistsF = new ExistsF {
      object Fint extends F[Int] {
        override def elim: Prod[Int, Int=>Int] = Prod(i, j => j)
      }
      override def elim[B](f: Helper[B]): B = f.elim[Int](Fint)
    }
  }
  
}